import re
import time
import os
from datetime import datetime, timedelta
import pytz
import atexit
from collections import deque
import hashlib
import base64
import json
from cryptography.fernet import Fernet
import subprocess
import uuid
import getpass
import sys
import requests
import socket

RESET = "\033[0m"
VERDE = "\033[1;32m"
AZUL = "\033[1;34m"
AMARELO = "\033[1;33m"
LARANJA = "\033[38;5;208m"
VERMELHO = "\033[1;31m"
CINZA = "\033[1;30m"
ROXO = "\033[1;35m"

LOG_ARQUIVO = "/var/www/html/log-output.txt"
HEARTBEAT_FILE = "/var/www/html/fortigate_heartbeat.txt"
TEMPO_MINIMO_ENTRE_TENTATIVAS = 5
LOG_FILE = "/var/log/syslog"

# URL do servidor de licenciamento
LICENSE_SERVER = "https://securelogtechonsi.ddns.net/license_manager.php"
VALIDATION_SERVER = "https://securelogtechonsi.ddns.net/validate_license.php"

# Arquivo local para cache da licença
LICENSE_CACHE_FILE = "/var/www/html/.securelog_license_cache.dat"

def obter_ip_publico():
    """Obtém o IP público do usuário"""
    try:
        response = requests.get('https://api.ipify.org', timeout=10)
        if response.status_code == 200:
            return response.text.strip()
    except:
        pass
    
    # Fallback: tenta outros serviços
    services = [
        'https://checkip.amazonaws.com',
        'https://icanhazip.com',
        'https://ifconfig.me/ip'
    ]
    
    for service in services:
        try:
            response = requests.get(service, timeout=5)
            if response.status_code == 200:
                return response.text.strip()
        except:
            continue
    
    return None

def obter_ip_local():
    """Obtém o IP local como fallback"""
    try:
        # Conecta a um socket para descobrir o IP local
        with socket.socket(socket.AF_INET, socket.SOCK_DGRAM) as s:
            s.connect(("8.8.8.8", 80))
            return s.getsockname()[0]
    except:
        return "127.0.0.1"

class ServerLicenseManager:
    def __init__(self):
        self.license_cache_file = LICENSE_CACHE_FILE
        self.public_ip = None
        self.local_ip = None

    def obter_ips(self):
        """Obtém IP público e local"""
        if not self.public_ip:
            self.public_ip = obter_ip_publico()
            self.local_ip = obter_ip_local()
        return self.public_ip, self.local_ip

    def solicitar_licenca_trial(self):
        """Solicita uma licença trial do servidor"""
        try:
            public_ip, local_ip = self.obter_ips()
            
            if not public_ip:
                print(f"{VERMELHO}✗ Não foi possível obter o IP público{RESET}")
                return False

            payload = {
                'action': 'request_trial',
                'public_ip': public_ip,
                'local_ip': local_ip,
                'hostname': socket.gethostname(),
                'timestamp': datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                'mac_address': self.obter_mac_address()
            }

            response = requests.post(
                LICENSE_SERVER,
                json=payload,
                timeout=1
            )

            if response.status_code == 200:
                result = response.json()
                if result.get('success'):
                    license_data = result.get('license', {})
                    # Salva licença em cache
                    self.salvar_licenca_cache(license_data)
                    print(f"{VERDE}✓ Trial ativado com sucesso!{RESET}")
                    print(f"{AZUL}IP: {public_ip}{RESET}")
                    print(f"{AZUL}Expira em: {license_data.get('expiry_date', 'N/A')}{RESET}")
                    return True
                else:
                    error_msg = result.get('message', 'Erro desconhecido')
                    print(f"{VERMELHO}✗ Falha ao ativar trial: {error_msg}{RESET}")
                    return False
            else:
                print(f"{VERMELHO}✗ Erro de conexão com servidor{RESET}")
                return False

        except requests.exceptions.Timeout:
            print(f"{VERMELHO}✗ Timeout ao conectar com servidor{RESET}")
            return False
        except Exception as e:
            print(f"{VERMELHO}✗ Erro ao solicitar trial: {str(e)}{RESET}")
            return False

    def validar_licenca_servidor(self):
        """Valida a licença com o servidor remoto"""
        try:
            public_ip, local_ip = self.obter_ips()
            
            # Primeiro tenta validar com cache
            cache_data = self.carregar_licenca_cache()
            if cache_data:
                license_key = cache_data.get('license_key')
            else:
                return False, "Licença não encontrada"

            payload = {
                'action': 'validate_license',
                'license_key': license_key,
                'public_ip': public_ip,
                'local_ip': local_ip,
                'timestamp': datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            }

            response = requests.post(
                VALIDATION_SERVER,
                json=payload,
                timeout=10
            )

            if response.status_code == 200:
                result = response.json()
                if result.get('valid'):
                    # Atualiza cache se necessário
                    if 'license' in result:
                        self.salvar_licenca_cache(result['license'])
                    return True, result.get('message', 'Licença válida')
                else:
                    return False, result.get('message', 'Licença inválida')
            else:
                return False, "Erro de conexão com servidor"

        except Exception as e:
            # Em caso de erro de conexão, usa cache como fallback
            cache_data = self.carregar_licenca_cache()
            if cache_data:
                if self.verificar_validade_cache(cache_data):
                    print(f"{LARANJA}⚠ Usando cache (offline){RESET}")
                    return True, "Licença válida (cache)"
            return False, f"Erro na validação: {str(e)}"

    def salvar_licenca_cache(self, license_data):
        """Salva licença em cache local"""
        try:
            cache_data = {
                'license_key': license_data.get('license_key'),
                'public_ip': license_data.get('public_ip'),
                'expiry_date': license_data.get('expiry_date'),
                'issue_date': license_data.get('issue_date'),
                'type': license_data.get('type', 'trial'),
                'last_validation': datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            }
            
            with open(self.license_cache_file, 'w') as f:
                json.dump(cache_data, f)
                
        except Exception as e:
            print(f"{VERMELHO}Erro ao salvar cache: {e}{RESET}")

    def carregar_licenca_cache(self):
        """Carrega licença do cache local"""
        try:
            if os.path.exists(self.license_cache_file):
                with open(self.license_cache_file, 'r') as f:
                    return json.load(f)
            return None
        except:
            return None

    def verificar_validade_cache(self, cache_data):
        """Verifica se o cache ainda é válido"""
        try:
            expiry_str = cache_data.get('expiry_date')
            if not expiry_str:
                return False

            expiry_date = datetime.strptime(expiry_str, "%Y-%m-%d %H:%M:%S")
            return datetime.now() < expiry_date

        except:
            return False

    def obter_mac_address(self):
        """Obtém MAC address da interface primária"""
        try:
            result = subprocess.run(['ip', 'route', 'show', 'default'], 
                                  capture_output=True, text=True)
            if result.returncode == 0:
                interface_match = re.search(r'dev\s+(\w+)', result.stdout)
                if interface_match:
                    interface = interface_match.group(1)
                    mac_result = subprocess.run(['cat', f'/sys/class/net/{interface}/address'],
                                              capture_output=True, text=True)
                    if mac_result.returncode == 0 and mac_result.stdout.strip():
                        return mac_result.stdout.strip()
        except:
            pass
        return "unknown"

    def dias_restantes(self):
        """Calcula dias restantes do trial"""
        cache_data = self.carregar_licenca_cache()
        if not cache_data:
            return 0

        try:
            expiry_str = cache_data.get('expiry_date')
            if not expiry_str:
                return 0

            expiry_date = datetime.strptime(expiry_str, "%Y-%m-%d %H:%M:%S")
            dias = (expiry_date - datetime.now()).days
            return max(0, dias)

        except:
            return 0

    def informacoes_licenca(self):
        """Retorna informações da licença"""
        return self.carregar_licenca_cache()

def verificar_licenca_obrigatoria():
    """Verificação obrigatória da licença antes de executar"""
    license_manager = ServerLicenseManager()
    
    print(f"{AZUL}")
    print("╔══════════════════════════════════════════════════════════════╗")
    print("║                   SECURELOG - VALIDAÇÃO                      ║")
    print("╚══════════════════════════════════════════════════════════════╝")
    print(f"{RESET}")

    # Verifica se já tem licença cacheada
    cache_data = license_manager.carregar_licenca_cache()
    if cache_data and license_manager.verificar_validade_cache(cache_data):
        print(f"{VERDE}✓ Licença trial válida encontrada{RESET}")
        dias_restantes = license_manager.dias_restantes()
        print(f"{AZUL}✓ Dias restantes: {dias_restantes}{RESET}")
        return True

    # Se não tem licença válida, solicita ao servidor
    print(f"{AMARELO}Solicitando licença trial...{RESET}")
    
    public_ip, local_ip = license_manager.obter_ips()
    print(f"{AZUL}IP Público: {public_ip or 'Não detectado'}{RESET}")
    print(f"{AZUL}IP Local: {local_ip}{RESET}")

    if license_manager.solicitar_licenca_trial():
        return True
    else:
        print(f"{VERMELHO}")
        print("╔══════════════════════════════════════════════════════════════╗")
        print("║               FALHA NA ATIVAÇÃO DO TRIAL                     ║")
        print("║                                                              ║")
        print("║  Não foi possível ativar o trial. Possíveis causas:          ║")
        print("║  • Sem conexão com a internet                                ║")
        print("║  • Já utilizou trial anteriormente                           ║")
        print("║  • Limite de trials esgotado                                 ║")
        print("║  • SecureLog temporariamente indisponível                    ║")
        print("║                                                              ║")
        print("║  Contato: Diego Maciel - TechON S.I                          ║")
        print("║  Telefone: +55 37 9 9130-5575                                ║")
        print("╚══════════════════════════════════════════════════════════════╝")
        print(f"{RESET}")
        return False

def validar_licenca_periodicamente():
    """Valida licença periodicamente durante execução"""
    license_manager = ServerLicenseManager()
    
    # Primeiro tenta com cache
    cache_data = license_manager.carregar_licenca_cache()
    if cache_data and license_manager.verificar_validade_cache(cache_data):
        return True

    # Se cache inválido, valida com servidor
    valido, mensagem = license_manager.validar_licenca_servidor()
    if not valido:
        print(f"{VERMELHO}✗ Validação periódica falhou: {mensagem}{RESET}")
    return valido

def obter_devname_firewall():
    """Extrai o devname do firewall dos logs do syslog"""
    try:
        devnames_encontrados = []
        
        # Estratégia 1: Procura nos logs recentes
        with open(LOG_FILE, 'r', encoding='utf-8', errors='ignore') as f:
            linhas = f.readlines()
            # Analisa as últimas 1000 linhas (logs mais recentes)
            for linha in reversed(linhas[-1000:]):
                if 'devname=' in linha:
                    devname_match = re.search(r'devname="([^"]+)"', linha)
                    if devname_match:
                        devname = devname_match.group(1)
                        if devname and devname != "unknown" and devname.strip():
                            devnames_encontrados.append(devname)
        
        # Estratégia 2: Se não encontrou, procura em todo o arquivo
        if not devnames_encontrados:
            with open(LOG_FILE, 'r', encoding='utf-8', errors='ignore') as f:
                for linha in f:
                    if 'devname=' in linha:
                        devname_match = re.search(r'devname="([^"]+)"', linha)
                        if devname_match:
                            devname = devname_match.group(1)
                            if devname and devname != "unknown" and devname.strip():
                                devnames_encontrados.append(devname)
                                break  # Para no primeiro válido
        
        # Retorna o devname mais frequente ou o primeiro encontrado
        if devnames_encontrados:
            from collections import Counter
            contador = Counter(devnames_encontrados)
            return contador.most_common(1)[0][0]
        else:
            return "Firewall_Desconhecido"
            
    except Exception as e:
        print(f"{VERMELHO}Erro ao obter devname: {e}{RESET}")
        return "Firewall_Desconhecido"

class RateLimiter:
    def __init__(self, max_events, time_window):
        self.max_events = max_events
        self.time_window = time_window
        self.events = deque()

    def allow(self):
        now = time.time()
        while self.events and now - self.events[0] > self.time_window:
            self.events.popleft()

        if len(self.events) < self.max_events:
            self.events.append(now)
            return True
        return False

def cleanup():
    """Remove o arquivo heartbeat quando o script é encerrado"""
    try:
        if os.path.exists(HEARTBEAT_FILE):
            os.remove(HEARTBEAT_FILE)
    except Exception as e:
        print(f"{VERMELHO}Erro ao limpar heartbeat: {e}{RESET}")

def data_brasilia():
    fuso = pytz.timezone("America/Sao_Paulo")
    return datetime.now(fuso).strftime("%d/%m/%Y")

def hora_brasilia():
    fuso = pytz.timezone("America/Sao_Paulo")
    return datetime.now(fuso).strftime("%H:%M:%S")

def tail_f(file_path):
    """Tail resiliente: detecta rotação de logs"""
    import os

    def open_log():
        f = open(file_path, "r", encoding="utf-8", errors="replace")
        f.seek(0, os.SEEK_END)
        return f, os.fstat(f.fileno()).st_ino

    f, inode = open_log()

    while True:
        line = f.readline()
        if not line:
            time.sleep(0.2)
            try:
                # Detecta se o inode mudou (logrotate)
                if os.stat(file_path).st_ino != inode:
                    f.close()
                    f, inode = open_log()
            except FileNotFoundError:
                time.sleep(1)
            continue
        yield line

def cor_metodo(metodo):
    metodo = metodo.upper()
    if metodo == "SSH":
        return f"{LARANJA}{metodo}{RESET}"
    elif metodo == "VPN":
        return f"{ROXO}{metodo}{RESET}"
    else:
        return f"{AMARELO}{metodo}{RESET}"

def atualizar_heartbeat():
    """Atualiza o arquivo heartbeat com o timestamp atual"""
    try:
        with open(HEARTBEAT_FILE, "w") as f:
            f.write(str(time.time()))
    except Exception as e:
        print(f"{VERMELHO}Erro ao atualizar heartbeat: {e}{RESET}")

def processar_linha_vpn(linha, ip_atual, hora):
    """Processa linhas relacionadas a VPN"""
    # Tentativa de login VPN falha
    if 'logdesc="SSL VPN login fail"' in linha or 'action="ssl-login-fail"' in linha:
        user_match = re.search(r'user="([^"]+)"', linha)
        ip_match = re.search(r'remip=([\d.]+)', linha)
        reason_match = re.search(r'reason="([^"]+)"', linha)

        user = user_match.group(1) if user_match else "N/A"
        ip = ip_match.group(1) if ip_match else ip_atual
        reason = reason_match.group(1) if reason_match else "motivo desconhecido"

        # Determina o tipo de falha
        reason_lower = reason.lower()
        if reason_lower in ["sslvpn_login_unknown_user", "no such user", "name_invalid"]:
            status_plain = "usuário inválido"
            status_color = f"{LARANJA}{status_plain}{RESET}"
        elif "permission" in reason_lower or "invalid" in reason_lower or "incorrect" in reason_lower:
            status_plain = "senha incorreta"
            status_color = f"{VERMELHO}{status_plain}{RESET}"
        else:
            status_plain = f"falha: {reason}"
            status_color = f"{AMARELO}{status_plain}{RESET}"

        # Registra o evento
        metodo_cor = cor_metodo("VPN")
        print(f"Data: {data} | Hora: {hora} | {VERDE}Método:{RESET} {metodo_cor} | {AZUL}User:{RESET} {user} | IP: {ip} | {VERDE}Status:{RESET} {status_color}")
        with open(LOG_ARQUIVO, "a") as f:
            f.write(f"Data: {data} | Hora: {hora} | Método: VPN | User: {user} | IP: {ip} | Status: {status_plain}\n")
        return True

    # 1. Conexão VPN estabelecida
    elif ('logdesc="SSL VPN tunnel up"' in linha or 'msg="SSL tunnel established"' in linha) and 'tunneltype="ssl-tunnel"' in linha:
        user_match = re.search(r'user="([^"]+)"', linha)
        ip_match = re.search(r'remip=([\d.]+)', linha)

        user = user_match.group(1) if user_match else "N/A"
        ip = ip_match.group(1) if ip_match else ip_atual

        status_plain = "Conexão VPN ativa"
        status_color = f"{VERDE}{status_plain}{RESET}"
        metodo_cor = cor_metodo("VPN")

        print(f"Data: {data} | Hora: {hora} | {VERDE}Método:{RESET} {metodo_cor} | {AZUL}User:{RESET} {user} | IP: {ip} | {VERDE}Status:{RESET} {status_color}")
        with open(LOG_ARQUIVO, "a") as f:
            f.write(f"Data: {data} | Hora: {hora} | Método: VPN | User: {user} | IP: {ip} | Status: {status_plain}\n")
        return True

    return False

def menu_renovacao():
    """Menu para renovação do trial"""
    license_manager = ServerLicenseManager()

    print(f"{AZUL}")
    print("╔══════════════════════════════════════════════════════════════╗")
    print("║                   RENOVAÇÃO DO SECURELOG                     ║")
    print("╚══════════════════════════════════════════════════════════════╝")
    print(f"{RESET}")

    info = license_manager.informacoes_licenca()
    if info:
        print(f"{VERDE}Licença atual:{RESET}")
        print(f"  Data de instalação: {info.get('issue_date', 'N/A')}")
        print(f"  Data de expiração: {info.get('expiry_date', 'N/A')}")
        print(f"  Dias restantes: {license_manager.dias_restantes()}")
        print(f"  IP: {info.get('public_ip', 'N/A')}")

    print(f"\n{AMARELO}Para renovar, entre em contato com o administrador.{RESET}")
    print(f"{AZUL}Você receberá um código de renovação.{RESET}")
    print("\n1 - Tentar reativar trial")
    print("2 - Voltar")

    opcao = input(f"\n{AMARELO}Opção: {RESET}").strip()

    if opcao == "1":
        if license_manager.solicitar_licenca_trial():
            print(f"{VERDE}✓ Trial reativado com sucesso! Reiniciando...{RESET}")
            time.sleep(3)
            # Reinicia o script para aplicar as mudanças
            os.execv(sys.executable, [sys.executable] + sys.argv)
        else:
            print(f"{VERMELHO}✗ Falha ao reativar trial{RESET}")
        input(f"\n{AMARELO}Pressione Enter para continuar...{RESET}")
    elif opcao == "2":
        return

def verificar_trial_obrigatorio():
    """Verificação obrigatória da licença antes de qualquer execução"""
    if not verificar_licenca_obrigatoria():
        print(f"{VERMELHO}Encerrando SecureLog...{RESET}")
        time.sleep(3)
        exit(1)
    
    license_manager = ServerLicenseManager()
    dias_restantes = license_manager.dias_restantes()
    
    print(f"{VERDE}✓ SecureLog ativado com sucesso!{RESET}")
    print(f"{AZUL}✓ Dias restantes do trial: {dias_restantes}{RESET}")
    
    if dias_restantes <= 3:
        print(f"{LARANJA}⚠ Trial expira em {dias_restantes} dias! Entre em contato para renovação.{RESET}")
    
    time.sleep(3)
    return True

def mostrar_banner():
    os.system("clear")
    
    # Obtém o devname atual
    devname = obter_devname_firewall()
    
    print(f"{AZUL}")
    print("███████╗███████╗ ██████╗██╗   ██╗██████╗ ███████╗██╗      ██████╗ ██████╗ ")
    print("██╔════╝██╔════╝██╔════╝██║   ██║██╔══██╗██╔════╝██║     ██╔════╝██╔════╝ ")
    print("███████╗█████╗  ██║     ██║   ██║██████╔╝█████╗  ██║     ██║     ██║  ███╗")
    print("╚════██║██╔══╝  ██║     ██║   ██║██╔══██╗██╔══╝  ██║     ██║     ██║   ██║")
    print("███████║███████╗╚██████╗╚██████╔╝██║  ██║███████╗███████╗╚██████╗╚██████╔╝")
    print("╚══════╝╚══════╝ ╚═════╝ ╚═════╝ ╚═╝  ╚═╝╚══════╝╚══════╝ ╚═════╝ ╚═════╝ ")
    
    license_manager = ServerLicenseManager()
    dias_restantes = license_manager.dias_restantes()
    info = license_manager.informacoes_licenca()
    ip_publico = info.get('public_ip', 'N/A') if info else 'N/A'
    
    print(f"{CINZA}                 SecureLog - FortiGate - Trial ({dias_restantes} dias restantes){RESET}")
    print(f"{CINZA}                  Firewall: {devname}{RESET}")
    print(f"{CINZA}                  IP: {ip_publico} - © 2025 By: Diêgo Maciel - TechON S.I{RESET}")

    print(f"\n{AMARELO}Opções:{RESET}")
    print(f"{VERDE}[1]{RESET} Iniciar monitoramento")
    print(f"{VERDE}[2]{RESET} Renovar trial")
    print(f"{VERDE}[3]{RESET} Sair")

    opcao = input(f"\n{AZUL}Selecione uma opção: {RESET}").strip()

    if opcao == "2":
        menu_renovacao()
        return False
    elif opcao == "3":
        exit(0)

    return True

# Execução principal
if __name__ == "__main__":
    # Verificação OBRIGATÓRIA da licença
    verificar_trial_obrigatorio()

    # Mostra banner e menu
    if not mostrar_banner():
        # Se saiu do menu de renovação, recarrega o script
        os.execv(sys.executable, [sys.executable] + sys.argv)

    # Inicialização normal
    atexit.register(cleanup)
    
    # Cria/carrega arquivo de heartbeat
    with open(HEARTBEAT_FILE, "w") as f:
        f.write(str(time.time()))

    # Variáveis de estado
    ultimos_metodos_por_ip = {}
    ultimo_ip_geral = None
    ultimo_metodo_geral = None
    ultimo_heartbeat = time.time()
    INTERVALO_HEARTBEAT = 30

    ultima_validacao = time.time()
    INTERVALO_VALIDACAO = 300  # 5 minutos

    # Loop principal
    for linha in tail_f(LOG_FILE):
        data = data_brasilia()
        hora = hora_brasilia()

        # Valida licença periodicamente
        if time.time() - ultima_validacao >= INTERVALO_VALIDACAO:
            if not validar_licenca_periodicamente():
                print(f"{VERMELHO}╔══════════════════════════════════════════════════════════════╗{RESET}")
                print(f"{VERMELHO}║                 LICENÇA INVÁLIDA - PARANDO                  ║{RESET}")
                print(f"{VERMELHO}║      Licença revogada ou expirada. Encerrando monitor.      ║{RESET}")
                print(f"{VERMELHO}╚══════════════════════════════════════════════════════════════╝{RESET}")
                exit(1)
            ultima_validacao = time.time()

        # Atualiza heartbeat a cada 30 segundos
        if time.time() - ultimo_heartbeat >= INTERVALO_HEARTBEAT:
            atualizar_heartbeat()
            ultimo_heartbeat = time.time()

        # Extrai IP da linha
        ip_match = re.search(r'(\d{1,3}(?:\.\d{1,3}){3})', linha)
        if ip_match:
            ip_atual = ip_match.group(1)
            ultimo_ip_geral = ip_atual
        else:
            ip_atual = ultimo_ip_geral

        # Ignora linhas sem IP válido
        if not ip_atual or ip_atual == "N/A":
            continue

        # 1. Processa eventos VPN
        if 'subtype="vpn"' in linha.lower():
            if processar_linha_vpn(linha, ip_atual, hora):
                continue

        # 2. Processa bloqueios brute force
        if "Login disabled from IP" in linha or "blocked" in linha.lower():
            match = re.search(r'IP ([\d.]+)', linha)
            if match:
                ip = match.group(1)
                segundos_match = re.search(r'for (\d+) seconds', linha)
                segundos = segundos_match.group(1) if segundos_match else "N/A"
                status_plain = f"bloqueado por brute force ({segundos}s)"
                status_color = f"{VERMELHO}{status_plain}{RESET}"

                if "sshd" in linha.lower() or "ssh" in linha.lower():
                    metodo = "SSH"
                else:
                    metodo = ultimos_metodos_por_ip.get(ip, ultimo_metodo_geral or "HTTPS")

                metodo_cor = cor_metodo(metodo)
                print(f"Data: {data} | Hora: {hora} | {VERDE}Método:{RESET} {metodo_cor} | IP: {ip} | {VERMELHO}Status:{RESET} {status_color}")
                with open(LOG_ARQUIVO, "a") as f:
                    f.write(f"Data: {data} | Hora: {hora} | Método: {metodo} | IP: {ip} | Status: {status_plain}\n")
            continue

        # 3. Processa tentativas SSH
        if "sshd" in linha.lower() and ("failed password" in linha.lower() or "accepted password" in linha.lower()):
            match = re.search(r'from\s+([\d.]+)', linha)
            if match:
                ip = match.group(1)
                if "accepted password" in linha.lower():
                    status_plain = "login SSH bem-sucedido"
                    status_color = f"{VERDE}{status_plain}{RESET}"
                else:
                    status_plain = "senha incorreta"
                    status_color = f"{VERMELHO}{status_plain}{RESET}"

                metodo = "SSH"
                user_match = re.search(r'for (\w+) from', linha)
                user = user_match.group(1) if user_match else "N/A"

                ultimos_metodos_por_ip[ip] = metodo
                ultimo_metodo_geral = metodo

                metodo_cor = cor_metodo(metodo)
                print(f"Data: {data} | Hora: {hora} | {VERDE}Método:{RESET} {metodo_cor} | {AZUL}User:{RESET} {user} | IP: {ip} | {VERDE}Status:{RESET} {status_color}")
                with open(LOG_ARQUIVO, "a") as f:
                    f.write(f"Data: {data} | Hora: {hora} | Método: {metodo} | User: {user} | IP: {ip} | Status: {status_plain}\n")
            continue

        # 4. Processa logs do FortiGate (HTTPS)
        if 'action="login"' in linha and ('status="failed"' in linha or 'status="success"' in linha):
            user_match = re.search(r'user="([^"]+)"', linha)
            ip_match = re.search(r'srcip=([\d.]+)', linha)
            status_match = re.search(r'status="([^"]+)"', linha)
            reason_match = re.search(r'reason="([^"]+)"', linha)
            method_match = re.search(r'method="([^"]+)"', linha)

            user = user_match.group(1) if user_match else "N/A"
            ip = ip_match.group(1) if ip_match else ip_atual
            status_val = status_match.group(1) if status_match else "N/A"
            reason_val = reason_match.group(1) if reason_match else ""
            metodo = method_match.group(1).upper() if method_match else "HTTPS"

            if ip == "N/A":
                continue

            if ip != "N/A":
                ultimos_metodos_por_ip[ip] = metodo
                ultimo_metodo_geral = metodo

            if 'action="logout"' in linha:
                continue

            metodo_cor = cor_metodo(metodo)

            if status_val == "failed" and reason_val == "passwd_invalid":
                status_color = f"{VERMELHO}senha incorreta{RESET}"
                status_plain = "senha incorreta"
            elif status_val == "failed" and reason_val == "name_invalid":
                status_color = f"{VERMELHO}usuário inválido{RESET}"
                status_plain = "usuário inválido"
            elif status_val == "success":
                status_color = f"{VERDE}sucesso{RESET}"
                status_plain = "sucesso"
            else:
                continue

            print(f"Data: {data} | Hora: {hora} | {VERDE}Método:{RESET} {metodo_cor} | {AZUL}User:{RESET} {user} | IP: {ip} | {VERDE}Status:{RESET} {status_color}")
            with open(LOG_ARQUIVO, "a") as f:
                f.write(f"Data: {data} | Hora: {hora} | Método: {metodo} | User: {user} | IP: {ip} | Status: {status_plain}\n")
            continue
