<?php
session_start();

if (!isset($_SESSION['logado']) || $_SESSION['logado'] !== true) {
  header('Location: index.html');
  exit;
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
  <meta charset="UTF-8">
  <title>SecureLog TRIAL</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">

  <link rel="icon" type="image/png" href="logo.png">
  <link rel="shortcut icon" href="logo.png">
  <link rel="apple-touch-icon" href="logo.png">

  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <style>
    :root {
      --primary: #EE3124; /* Fortinet Red */
      --primary-dark: #C00;
      --dark: #121212;
      --darker: #0A0A0A;
      --dark-gray: #1E1E1E;
      --medium-gray: #2D2D2D;
      --light-gray: #E0E0E0;
      --lighter-gray: #F5F5F5;
      --success: #00A651;
      --danger: #EE3124;
      --info: #0066CC;
      --ssh: #F04E23;
      --https: #0066CC;
      --vpn: #AA00FF;
      --text-primary: #FFFFFF;
      --text-secondary: #B0B0B0;
      --text-tertiary: #808080;
      --shadow-sm: 0 1px 3px rgba(0, 0, 0, 0.12);
      --shadow-md: 0 4px 6px rgba(0, 0, 0, 0.15);
      --shadow-lg: 0 10px 25px rgba(0, 0, 0, 0.2);
      --border-radius: 8px;
      --border-radius-sm: 4px;
      --transition: all 0.25s cubic-bezier(0.4, 0, 0.2, 1);
    }

    * {
      box-sizing: border-box;
      margin: 0;
      padding: 0;
    }

    body {
      background-color: var(--dark);
      color: var(--text-primary);
      font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
      line-height: 1.6;
      -webkit-font-smoothing: antialiased;
      -moz-osx-font-smoothing: grayscale;
    }

    /* Layout Structure */
    .app-container {
      display: grid;
      grid-template-rows: auto 1fr;
      min-height: 100vh;
    }

    /* Header */
    .app-header {
      background-color: var(--darker);
      padding: 0 2rem;
      height: 64px;
      display: flex;
      align-items: center;
      justify-content: space-between;
      border-bottom: 1px solid rgba(255, 255, 255, 0.05);
      box-shadow: var(--shadow-md);
      position: sticky;
      top: 0;
      z-index: 100;
    }

    .logo-container {
      display: flex;
      align-items: center;
      gap: 1rem;
    }

    .logo-img {
      height: 28px;
    }

    .app-title {
      font-size: 1.25rem;
      font-weight: 600;
      color: var(--text-primary);
      margin: 0;
    }

    .app-version {
      font-size: 0.85rem;
      background: rgba(255, 255, 255, 0.05);
      padding: 0.25rem 0.5rem;
      border-radius: 6px;
      display: inline-block;
      margin-left: 0.5rem;
    }

    .version-green {
      color: #00ff88; /* Verde neon */
    }

    .by-blue {
      color: #00ccff; /* Azul claro */
    }

    .link-yellow {
      color: #ffff66; /* Amarelo claro */
      text-decoration: underline;
    }

    .header-controls {
      display: flex;
      align-items: center;
      gap: 1rem;
    }

    /* Main Content */
    .app-main {
      padding: 2rem;
      display: grid;
      grid-template-columns: 1fr;
      gap: 1.5rem;
      max-width: 1920px;
      margin: 0 auto;
      width: 100%;
    }

    /* Dashboard Grid */
    .dashboard-grid {
      display: grid;
      grid-template-columns: repeat(12, 1fr);
      gap: 1.5rem;
    }

    /* Cards */
    .card {
      background-color: var(--dark-gray);
      border-radius: var(--border-radius);
      box-shadow: var(--shadow-sm);
      overflow: hidden;
      transition: var(--transition);
      border: 1px solid rgba(255, 255, 255, 0.05);
    }

    .card:hover {
      box-shadow: var(--shadow-md);
      transform: translateY(-2px);
    }

    .card-header {
      padding: 1rem 1.5rem;
      border-bottom: 1px solid rgba(255, 255, 255, 0.05);
      display: flex;
      justify-content: space-between;
      align-items: center;
    }

    .card-title {
      font-size: 1rem;
      font-weight: 600;
      color: var(--text-primary);
      margin: 0;
      display: flex;
      align-items: center;
      gap: 0.5rem;
    }

    .card-title i {
      color: var(--primary);
      font-size: 0.9rem;
    }

    .card-controls {
      display: flex;
      gap: 0.5rem;
    }

    .card-body {
      padding: 1.5rem;
    }

    /* Stats Overview */
    .stats-overview {
      grid-column: span 12;
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
      gap: 1rem;
    }

    .stat-card {
      background-color: var(--medium-gray);
      border-radius: var(--border-radius);
      padding: 1.25rem;
      display: flex;
      flex-direction: column;
      align-items: flex-start;
    }

    .stat-label {
      font-size: 0.875rem;
      color: var(--text-secondary);
      margin-bottom: 0.5rem;
      display: flex;
      align-items: center;
      gap: 0.5rem;
    }

    .stat-value {
      font-size: 1.75rem;
      font-weight: 600;
      color: var(--text-primary);
    }

    .stat-trend {
      font-size: 0.75rem;
      margin-top: 0.5rem;
      display: flex;
      align-items: center;
      gap: 0.25rem;
    }

    .trend-up {
      color: var(--success);
    }

    .trend-down {
      color: var(--danger);
    }

    .trend-neutral {
      color: var(--text-tertiary);
    }

    /* Charts */
    .chart-container {
      position: relative;
      height: 100%;
      min-height: 300px;
    }

    .chart-card {
      grid-column: span 4;
    }

    @media (max-width: 1400px) {
      .chart-card {
        grid-column: span 6;
      }
    }

    @media (max-width: 768px) {
      .chart-card {
        grid-column: span 12;
      }
    }

    /* Status Card */
    .status-card {
      grid-column: span 4;
      display: flex;
      flex-direction: column;
    }

    .status-container {
      flex: 1;
      display: flex;
      flex-direction: column;
      align-items: center;
      justify-content: center;
      padding: 2rem;
      text-align: center;
    }

    .status-indicator {
      width: 80px;
      height: 80px;
      border-radius: 50%;
      margin-bottom: 1.5rem;
      display: flex;
      align-items: center;
      justify-content: center;
      position: relative;
    }

    .status-indicator:before {
      content: '';
      position: absolute;
      inset: 0;
      border-radius: 50%;
      padding: 4px;
      background: linear-gradient(135deg, currentColor, transparent);
      -webkit-mask: linear-gradient(#fff 0 0) content-box, linear-gradient(#fff 0 0);
      -webkit-mask-composite: xor;
      mask-composite: exclude;
      pointer-events: none;
    }

    .status-active {
      background-color: rgba(0, 166, 81, 0.1);
      color: var(--success);
    }

    .status-inactive {
      background-color: rgba(238, 49, 36, 0.1);
      color: var(--danger);
    }

    .status-icon {
      font-size: 2rem;
    }

    .status-text {
      font-size: 1.25rem;
      font-weight: 600;
      margin-bottom: 0.5rem;
    }

    .status-time {
      font-size: 0.875rem;
      color: var(--text-secondary);
      max-width: 280px;
    }

    /* Logs Panel */
    .logs-panel {
      grid-column: span 12;
    }

    .logs-container {
      height: 500px;
      overflow-y: auto;
      background-color: var(--dark);
      border-radius: var(--border-radius-sm);
      padding: 1rem;
      font-family: 'SF Mono', 'Roboto Mono', monospace;
      font-size: 0.875rem;
      line-height: 1.7;
      white-space: pre-wrap;
    }

    /* Log highlighting */
    .log-entry {
      margin-bottom: 0.5rem;
    }

    .timestamp {
      color: var(--text-tertiary);
    }

    .success {
      color: var(--success);
      font-weight: 500;
    }

    .error {
      color: var(--danger);
      font-weight: 500;
    }

    .ssh {
      color: var(--ssh);
    }

    .https {
      color: var(--https);
    }

    .vpn {
      color: var(--vpn);
    }

    .info {
      color: var(--info);
    }

    /* Buttons */
    .btn {
      display: inline-flex;
      align-items: center;
      justify-content: center;
      gap: 0.5rem;
      padding: 0.5rem 1rem;
      border-radius: var(--border-radius-sm);
      font-size: 0.875rem;
      font-weight: 500;
      line-height: 1;
      cursor: pointer;
      transition: var(--transition);
      border: 1px solid transparent;
      background-color: var(--medium-gray);
      color: var(--text-primary);
    }

    .btn-sm {
      padding: 0.375rem 0.75rem;
      font-size: 0.8125rem;
    }

    .btn-primary {
      background-color: var(--primary);
      color: white;
    }

    .btn-primary:hover {
      background-color: var(--primary-dark);
    }

    .btn-outline {
      background-color: transparent;
      border-color: rgba(255, 255, 255, 0.15);
    }

    .btn-outline:hover {
      background-color: rgba(255, 255, 255, 0.05);
    }

    .btn-danger {
      background-color: var(--danger);
      color: white;
    }

    .btn-danger:hover {
      background-color: #C00;
    }

    .btn-success {
      background-color: var(--success);
      color: white;
    }

    .btn-success:hover {
      background-color: #008A45;
    }

    /* Badges */
    .badge {
      display: inline-flex;
      align-items: center;
      padding: 0.25rem 0.5rem;
      border-radius: 50px;
      font-size: 0.75rem;
      font-weight: 500;
      line-height: 1;
    }

    .badge-success {
      background-color: rgba(0, 166, 81, 0.15);
      color: var(--success);
    }

    .badge-danger {
      background-color: rgba(238, 49, 36, 0.15);
      color: var(--danger);
    }

    /* Toolbar */
    .toolbar {
      display: flex;
      justify-content: space-between;
      align-items: center;
      padding: 0.75rem 1.5rem;
      background-color: var(--medium-gray);
      border-bottom: 1px solid rgba(255, 255, 255, 0.05);
    }

    .toolbar-group {
      display: flex;
      align-items: center;
      gap: 0.75rem;
    }

    /* Refresh Info */
    .refresh-info {
      font-size: 0.8125rem;
      color: var(--text-tertiary);
      display: flex;
      align-items: center;
      gap: 0.5rem;
    }

    .refresh-status {
      display: flex;
      align-items: center;
      gap: 0.25rem;
    }

    .refresh-active {
      color: var(--success);
    }

    /* Trial Status Colors */
    .status-warning {
      background-color: rgba(255, 193, 7, 0.1);
      color: #FFC107;
    }

    .status-expired {
      background-color: rgba(238, 49, 36, 0.1);
      color: var(--danger);
    }

    .status-valid {
      background-color: rgba(0, 166, 81, 0.1);
      color: var(--success);
    }

    /* Trial Progress Bar */
    .trial-progress {
      width: 100%;
      height: 6px;
      background-color: rgba(255, 255, 255, 0.1);
      border-radius: 3px;
      margin: 1rem 0;
      overflow: hidden;
    }

    .trial-progress-bar {
      height: 100%;
      border-radius: 3px;
      transition: width 0.3s ease;
    }

    .progress-valid {
      background-color: var(--success);
    }

    .progress-warning {
      background-color: #FFC107;
    }

    .progress-expired {
      background-color: var(--danger);
    }

    /* Utility Classes */
    .text-muted {
      color: var(--text-tertiary);
    }

    .text-success {
      color: var(--success);
    }

    .text-danger {
      color: var(--danger);
    }

    .text-info {
      color: var(--info);
    }

    .mt-auto {
      margin-top: auto;
    }

    /* Scrollbar */
    ::-webkit-scrollbar {
      width: 8px;
      height: 8px;
    }

    ::-webkit-scrollbar-track {
      background: rgba(255, 255, 255, 0.05);
    }

    ::-webkit-scrollbar-thumb {
      background: rgba(255, 255, 255, 0.1);
      border-radius: 4px;
    }

    ::-webkit-scrollbar-thumb:hover {
      background: rgba(255, 255, 255, 0.15);
    }

    /* Animations */
    @keyframes pulse {
      0%, 100% {
        opacity: 1;
      }
      50% {
        opacity: 0.5;
      }
    }

    .pulse {
      animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
    }

    /* Responsive Adjustments */
    @media (max-width: 1024px) {
      .app-main {
        padding: 1.5rem;
      }
    }

    @media (max-width: 768px) {
      .app-header {
        padding: 0 1rem;
      }

      .app-main {
        padding: 1rem;
        gap: 1rem;
      }

      .dashboard-grid {
        grid-template-columns: 1fr;
      }

      .stats-overview {
        grid-template-columns: repeat(2, 1fr);
      }
    }

    @media (max-width: 480px) {
      .stats-overview {
        grid-template-columns: 1fr;
      }

      .header-controls {
        display: none;
      }
    }
  </style>
</head>
<body>
  <div class="app-container">
    <!-- Header -->
    <header class="app-header">
      <div class="logo-container">
        <img src="logo.png" alt="FortiGate Logo" class="logo-img" style="width: 70px; height: auto;">

        <span class="logo-text" style="font-size: 1.3rem; font-weight: bold;">
            TechOn <span style="color: var(--primary); font-size: 1.3rem;">S.I</span>
        </span>

        <span class="app-version version-green">Versão: 2.4.5 - Trial</span>
        <span class="app-version by-blue">By: Diêgo Maciel</span>
        <a href="https://securelogtechonsi.ddns.net/" class="app-version link-yellow" target="_blank">
        https://securelogtechonsi.ddns.net/
        </a>
      </div>
    </header>

    <!-- Main Content -->
    <main class="app-main">
      <div class="dashboard-grid">
        <!-- Stats Overview -->
        <div class="stats-overview">
          <div class="stat-card">
            <span class="stat-label">
              <i class="fas fa-shield-alt"></i> Total Eventos
            </span>
            <span class="stat-value" id="total-count">0</span>
            <span class="stat-trend trend-neutral">
            </span>
          </div>
          <div class="stat-card">
            <span class="stat-label">
              <i class="fas fa-check-circle"></i> Sucesso
            </span>
            <span class="stat-value success-count" id="success-count">0</span>
            <span class="stat-trend trend-neutral">
            </span>
          </div>
          <div class="stat-card">
            <span class="stat-label">
              <i class="fas fa-times-circle"></i> Erros
            </span>
            <span class="stat-value error-count" id="error-count">0</span>
            <span class="stat-trend trend-up">
            </span>
          </div>
          <div class="stat-card">
            <span class="stat-label">
              <i class="fas fa-terminal"></i> SSH
            </span>
            <span class="stat-value ssh-count" id="ssh-count">0</span>
            <span class="stat-trend trend-up">
            </span>
          </div>
          <div class="stat-card">
            <span class="stat-label">
              <i class="fas fa-lock"></i> HTTPS
            </span>
            <span class="stat-value https-count" id="https-count">0</span>
            <span class="stat-trend trend-neutral">
            </span>
          </div>
          <div class="stat-card">
            <span class="stat-label">
              <i class="fas fa-shield-alt"></i> VPN
            </span>
            <span class="stat-value vpn-count" id="vpn-count">0</span>
            <span class="stat-trend trend-neutral">
            </span>
          </div>
        </div>

        <!-- Charts Restantes -->
        <div class="card chart-card">
          <div class="card-header">
            <h2 class="card-title">
              <i class="fas fa-chart-bar"></i> Tentativas Diárias
            </h2>
          </div>
          <div class="card-body">
            <div class="chart-container">
              <canvas id="hourlyChart"></canvas>
            </div>
          </div>
        </div>

        <div class="card chart-card">
          <div class="card-header">
            <h2 class="card-title">
              <i class="fas fa-chart-line"></i> Tentativas Semanais
            </h2>
          </div>
          <div class="card-body">
            <div class="chart-container">
              <canvas id="dailyChart"></canvas>
            </div>
          </div>
        </div>

        <!-- Status Card -->
        <div class="card status-card">
          <div class="card-header">
            <h2 class="card-title">
              <i class="fas fa-shield-virus"></i> SecureLog Status
            </h2>
          </div>
          <div class="status-container" id="firewallStatus">
            <div class="status-indicator status-active">
              <i class="fas fa-check status-icon"></i>
            </div>
            <div class="status-text">ANALISANDO</div>
            <div class="status-time">Aguardando SecureLog...</div>
            <div class="mt-auto">
              <div class="refresh-info">
                <span>Ultima Atualização de pagina:</span>
                <span id="last-update">-</span>
              </div>
            </div>
          </div>
        </div>

        <!-- Trial Status Card -->
        <div class="card status-card">
          <div class="card-header">
            <h2 class="card-title">
              <i class="fas fa-clock"></i> Status do Trial
            </h2>
          </div>
          <div class="status-container" id="trialStatus">
            <div class="status-indicator status-active" id="trialIndicator">
              <i class="fas fa-clock status-icon" id="trialIcon"></i>
            </div>
            <div class="status-text" id="trialText">Carregando...</div>
            <div class="status-time" id="trialDays">Verificando licença</div>
            <div class="status-time" id="trialIP" style="font-size: 0.8rem; margin-top: 0.5rem;"></div>
            <div class="mt-auto">
              <button class="btn btn-outline btn-sm" id="renewBtn" style="display: none;">
                <i class="fas fa-sync-alt"></i> Renovar Trial
              </button>
            </div>
          </div>
        </div>

        <!-- Logs Panel -->
        <div class="card logs-panel">
          <div class="card-header">
            <h2 class="card-title">
              <i class="fas fa-terminal"></i> Logs de Autenticação
            </h2>
            <div class="card-controls">
              <div class="refresh-info">
                <span class="refresh-status refresh-active" id="refresh-status">
                  <i class="fas fa-circle-notch fa-spin"></i> Active
                </span>
              </div>
            </div>
          </div>
          <div class="toolbar">
            <div class="toolbar-group">
              <button class="btn btn-sm" id="pause-btn">
                <i class="fas fa-pause"></i> Pausar
              </button>
              <button class="btn btn-outline btn-sm" id="export-btn">
                <i class="fas fa-download"></i> Exportar
              </button>
              <button class="btn btn-outline btn-sm" id="reset-stats-btn">
                <i class="fas fa-redo"></i> Resetar Estatísticas
              </button>
            </div>
            <div class="toolbar-group">
              <button class="btn btn-outline btn-sm btn-danger" id="delete-logs-btn">
                <i class="fas fa-trash-alt"></i> Limpar Logs
              </button>
              <button class="btn btn-outline btn-sm" id="exit-btn">
                <i class="fas fa-sign-out-alt"></i> Sair
              </button>
            </div>
          </div>
          <div class="card-body" style="padding: 0;">
            <div class="logs-container" id="logs">Carregando logs...</div>
          </div>
        </div>
      </div>
    </main>
  </div>

  <!-- Bibliotecas para gráficos -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/moment"></script>
  <script src="https://cdn.jsdelivr.net/npm/chartjs-adapter-moment"></script>
  <script src="https://cdn.jsdelivr.net/npm/chartjs-plugin-datalabels@2.0.0"></script>

  <script>
  // Elementos da interface
  const logBox = document.getElementById("logs");
  const pauseBtn = document.getElementById("pause-btn");
  const exportBtn = document.getElementById("export-btn");
  const resetStatsBtn = document.getElementById("reset-stats-btn");
  const deleteLogsBtn = document.getElementById("delete-logs-btn");
  const exitBtn = document.getElementById("exit-btn");
  const totalCount = document.getElementById("total-count");
  const successCount = document.getElementById("success-count");
  const errorCount = document.getElementById("error-count");
  const sshCount = document.getElementById("ssh-count");
  const httpsCount = document.getElementById("https-count");
  const vpnCount = document.getElementById("vpn-count");
  const refreshStatus = document.getElementById("refresh-status");
  const lastUpdate = document.getElementById("last-update");
  const firewallStatus = document.getElementById("firewallStatus");

  // Estado da aplicação
  let isPaused = false;
  let refreshInterval = 2000;
  let logData = "";
  let stats = {
    total: 0,
    success: 0,
    error: 0,
    ssh: 0,
    https: 0,
    vpn: 0
  };

  // Dados para gráficos
  let hourlyAttempts = Array(24).fill(0);
  let dailyAttempts = Array(7).fill(0);
  let ipAttempts = {};
  let successIpAttempts = {};
  let vpnIpAttempts = {};
  let lastLogTimestamp = Date.now();
  let lastLogCount = 0;
  let lastSavedLogLength = 0;
  let lastResetDate = null;

  // Funções de persistência
  function saveChartData() {
    const chartData = {
      hourlyAttempts,
      dailyAttempts,
      ipAttempts,
      successIpAttempts,
      vpnIpAttempts,
      stats,
      lastSavedLogLength: logData.length,
      lastResetDate: lastResetDate ? lastResetDate.toISOString() : new Date().toISOString()
    };

    fetch('save-chart-data.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(chartData)
    });
  }

  async function loadChartData() {
    try {
      const response = await fetch('load-chart-data.php');
      const chartData = await response.json();

      const now = new Date();
      const currentDay = now.getDay(); // 0 = Domingo
      const currentHour = now.getHours();
      const currentDate = now.getDate();

      if (chartData && chartData.hourlyAttempts) {
        const lastSavedDate = chartData.lastResetDate ? new Date(chartData.lastResetDate) : null;

        // Se não há data de reset anterior ou mudou o dia, faz o reset
        if (!lastSavedDate || lastSavedDate.getDate() !== currentDate) {
          console.log("Reset diário detectado - novo dia:", currentDate);
          hourlyAttempts = Array(24).fill(0);

          // Se for domingo (0) e hora entre 00:00-01:00, zera dados semanais
          if (currentDay === 0 && currentHour === 0) {
            console.log("Reset semanal detectado - domingo 00:00");
            dailyAttempts = Array(7).fill(0);
          } else {
            dailyAttempts = chartData.dailyAttempts || Array(7).fill(0);
          }

          ipAttempts = chartData.ipAttempts || {};
          successIpAttempts = chartData.successIpAttempts || {};
          vpnIpAttempts = chartData.vpnIpAttempts || {};
          stats = chartData.stats || {
            total: 0, success: 0, error: 0, ssh: 0, https: 0, vpn: 0
          };

          lastResetDate = now;
          lastSavedLogLength = chartData.lastSavedLogLength || 0;

          saveChartData(); // Salva o novo estado resetado
        } else {
          // Carrega os dados normalmente - mesmo dia
          hourlyAttempts = chartData.hourlyAttempts || Array(24).fill(0);
          dailyAttempts = chartData.dailyAttempts || Array(7).fill(0);
          ipAttempts = chartData.ipAttempts || {};
          successIpAttempts = chartData.successIpAttempts || {};
          vpnIpAttempts = chartData.vpnIpAttempts || {};
          stats = chartData.stats || {
            total: 0, success: 0, error: 0, ssh: 0, https: 0, vpn: 0
          };
          lastResetDate = lastSavedDate ? new Date(lastSavedDate) : now;
          lastSavedLogLength = chartData.lastSavedLogLength || 0;
        }
      } else {
        // Primeira execução
        lastResetDate = now;
      }
    } catch (e) {
      console.error("Erro ao carregar dados dos gráficos:", e);
    }
  }

  // Format timestamp
  function formatTimestamp() {
    const now = new Date();
    return now.toLocaleTimeString() + '.' + now.getMilliseconds().toString().padStart(3, '0');
  }

  // Update statistics display
  function updateStats() {
    totalCount.textContent = stats.total.toLocaleString();
    successCount.textContent = stats.success.toLocaleString();
    errorCount.textContent = stats.error.toLocaleString();
    sshCount.textContent = stats.ssh.toLocaleString();
    httpsCount.textContent = stats.https.toLocaleString();
    vpnCount.textContent = stats.vpn.toLocaleString();
    lastUpdate.textContent = formatTimestamp();
  }

  // Reset statistics
  function resetStats() {
    stats = {
      total: 0,
      success: 0,
      error: 0,
      ssh: 0,
      https: 0,
      vpn: 0
    };
    hourlyAttempts = Array(24).fill(0);
    dailyAttempts = Array(7).fill(0);
    ipAttempts = {};
    successIpAttempts = {};
    vpnIpAttempts = {};
    lastSavedLogLength = 0;
    lastResetDate = new Date();

    localStorage.removeItem('fortigateChartData');

    updateStats();
    updateCharts();
  }

  // Processar linha de log e atualizar estatísticas - CORRIGIDO
  function processLogLine(line) {
    if (!line.trim()) return;

    const now = new Date();
    const hour = now.getHours();
    const day = now.getDay();

    // Contagem geral - TODOS os logs incrementam o total
    stats.total++;

    // Verificar tipo de log e atualizar estatísticas
    // AGORA: Todas as tentativas (sucesso e erro) contam para os gráficos
    if (line.includes("Status: sucesso") ||
      line.includes("Status: Conexão VPN ativa") ||
      line.includes("Status: senha incorreta") ||
      line.includes("Status: usuário inválido") ||
      line.includes("Status: falha") ||
      line.includes("Status: bloqueado")) {

      // Incrementa ambos os gráficos para TODOS os tipos de tentativa
      hourlyAttempts[hour]++;
    dailyAttempts[day]++;

    // Atualiza estatísticas específicas
    if (line.includes("Status: sucesso") || line.includes("Status: Conexão VPN ativa")) {
      stats.success++;
    } else {
      stats.error++;
    }
      }

      if (line.includes("Método: SSH")) {
        stats.ssh++;
      }
      else if (line.includes("Método: HTTPS")) {
        stats.https++;
      }
      else if (line.includes("Método: VPN")) {
        stats.vpn++;
      }

      // Extrair IP para os gráficos de IPs
      const ipMatch = line.match(/IP: ([\d.]+)/);
      if (ipMatch) {
        const ip = ipMatch[1];

        ipAttempts[ip] = (ipAttempts[ip] || 0) + 1;

        if (line.includes("Status: sucesso")) {
          successIpAttempts[ip] = (successIpAttempts[ip] || 0) + 1;
        }

        if (line.includes("Status: Conexão VPN ativa")) {
          vpnIpAttempts[ip] = (vpnIpAttempts[ip] || 0) + 1;
        }
      }

      lastLogTimestamp = Date.now();
  }

  // Atualizar gráficos
  function updateCharts() {
    hourlyChart.data.datasets[0].data = hourlyAttempts;
    hourlyChart.update();

    dailyChart.data.datasets[0].data = dailyAttempts;
    dailyChart.update();

    saveChartData();
  }

  // Verificar se o script Python está rodando com base na existência do arquivo
  async function checkPythonScript() {
    try {
      const response = await fetch('fortigate_heartbeat.txt?t=' + Date.now(), { method: 'HEAD' });
      return response.ok; // true se o arquivo existir (status 200)
    } catch (error) {
      return false;
    }
  }

  // Verificar status do firewall baseado no script Python
  async function updateFirewallStatus() {
    const isActive = await checkPythonScript();

    const indicator = firewallStatus.querySelector('.status-indicator');
    const icon = firewallStatus.querySelector('.status-icon');
    const text = firewallStatus.querySelector('.status-text');
    const timeInfo = firewallStatus.querySelector('.status-time');

    if (isActive) {
      indicator.className = 'status-indicator status-active';
      icon.className = 'fas fa-check status-icon';
      text.textContent = 'OPERACIONAL';
      text.style.color = 'var(--success)';
      timeInfo.textContent = 'SecureLog em execução';
    } else {
      indicator.className = 'status-indicator status-inactive';
      icon.className = 'fas fa-times status-icon';
      text.textContent = 'INOPERANTE';
      text.style.color = 'var(--danger)';
      timeInfo.textContent = 'SecureLog não está em execução!';
    }
  }

  // Verificar status do trial
  async function checkTrialStatus() {
    try {
      const response = await fetch('check_trial.php?t=' + Date.now());
      const data = await response.json();

      updateTrialDisplay(data);
      return data;
    } catch (error) {
      console.error("Erro ao verificar trial:", error);
      updateTrialDisplay({
        status: 'error',
        dias_restantes: 0,
        expiracao: 'Erro ao carregar'
      });
    }
  }

  // Atualizar display do trial
  function updateTrialDisplay(trialData) {
    const indicator = document.getElementById('trialIndicator');
    const icon = document.getElementById('trialIcon');
    const text = document.getElementById('trialText');
    const days = document.getElementById('trialDays');
    const renewBtn = document.getElementById('renewBtn');

    // Criar barra de progresso se não existir
    let progressBar = document.getElementById('trialProgress');
    if (!progressBar) {
      const progressContainer = document.createElement('div');
      progressContainer.className = 'trial-progress';
      progressContainer.innerHTML = '<div id="trialProgress" class="trial-progress-bar"></div>';
      days.parentNode.insertBefore(progressContainer, days.nextSibling);
      progressBar = document.getElementById('trialProgress');
    }
    document.getElementById('trialIP').textContent = `IP: ${trialData.ip_publico || 'N/A'}`;


    if (trialData.status === 'valid') {
      const diasRestantes = trialData.dias_restantes;

      // Calcular porcentagem (15 dias = 100%)
      const totalDias = trialData.total_days || 15;
      const porcentagem = Math.max(0, (diasRestantes / totalDias) * 100);
      if (diasRestantes > 7) {
        // Trial válido - verde
        indicator.className = 'status-indicator status-valid';
        icon.className = 'fas fa-check status-icon';
        text.textContent = 'SECURELOG ATIVO';
        text.style.color = 'var(--success)';
        progressBar.className = 'trial-progress-bar progress-valid';
        renewBtn.style.display = 'none';
      } else if (diasRestantes > 3) {
        // Aviso - amarelo
        indicator.className = 'status-indicator status-warning';
        icon.className = 'fas fa-exclamation-triangle status-icon';
        text.textContent = 'SECURELOG EXPIRANDO';
        text.style.color = '#FFC107';
        progressBar.className = 'trial-progress-bar progress-warning';
        renewBtn.style.display = 'inline-flex';
      } else if (diasRestantes > 0) {
        // Crítico - laranja/vermelho
        indicator.className = 'status-indicator status-warning';
        icon.className = 'fas fa-exclamation-circle status-icon';
        text.textContent = 'SECURELOG CRÍTICO';
        text.style.color = '#FF9800';
        progressBar.className = 'trial-progress-bar progress-warning';
        renewBtn.style.display = 'inline-flex';
      } else {
        // Expirado - vermelho
        indicator.className = 'status-indicator status-expired';
        icon.className = 'fas fa-times status-icon';
        text.textContent = 'SECURELOG EXPIRADO';
        text.style.color = 'var(--danger)';
        progressBar.className = 'trial-progress-bar progress-expired';
        renewBtn.style.display = 'inline-flex';
      }

      progressBar.style.width = `${porcentagem}%`;
      days.textContent = `${diasRestantes} dias restantes (Expira: ${trialData.expiracao})`;

    } else if (trialData.status === 'expired') {
      // Trial expirado
      indicator.className = 'status-indicator status-expired';
      icon.className = 'fas fa-times status-icon';
      text.textContent = 'SECURELOG EXPIRADO';
      text.style.color = 'var(--danger)';
      days.textContent = `Expirado em: ${trialData.expiracao}`;
      progressBar.style.width = '0%';
      progressBar.className = 'trial-progress-bar progress-expired';
      renewBtn.style.display = 'inline-flex';

    } else {
      // Erro
      indicator.className = 'status-indicator status-expired';
      icon.className = 'fas fa-question status-icon';
      text.textContent = 'ERRO';
      text.style.color = 'var(--text-tertiary)';
      days.textContent = 'Não foi possível verificar a licença';
      progressBar.style.width = '0%';
      progressBar.className = 'trial-progress-bar progress-expired';
      renewBtn.style.display = 'none';
    }
  }

  // Função para renovação
  function setupRenewal() {
    const renewBtn = document.getElementById('renewBtn');
    renewBtn.addEventListener('click', function() {
      alert('Para renovar o SecureLog, entre em contato com o administrador.\n\nEmail: diegomaciel.linux@hotmail.com');
    });
  }

  // Highlight log content
  function highlightLogs(text) {
    return text
    .replace(/Status: sucesso/g, 'Status: <span class="success">sucesso</span>')
    .replace(/Status: senha incorreta/g, 'Status: <span class="error">senha incorreta</span>')
    .replace(/Status: usuário inválido/g, 'Status: <span class="error">usuário inválido</span>')
    .replace(/Status: falha/g, 'Status: <span class="error">falha</span>')
    .replace(/Status: bloqueado/g, 'Status: <span class="error">bloqueado</span>')
    .replace(/Status: Conexão VPN ativa/g, 'Status: <span class="success">Conexão VPN ativa</span>')
    .replace(/Método: SSH/g, '<span class="ssh">Método: SSH</span>')
    .replace(/Método: HTTPS/g, '<span class="https">Método: HTTPS</span>')
    .replace(/Método: VPN/g, '<span class="vpn">Método: VPN</span>')
  }

  // Fetch logs from server
  async function fetchLogs() {
    if (isPaused) return;

    try {
      const response = await fetch('log-output.txt?t=' + new Date().getTime());
      const text = await response.text();
      if (text !== logData) {
        // Se o arquivo estiver vazio
        if (text.trim().length === 0) {
          logBox.innerHTML = `[${formatTimestamp()}] Todos os logs foram apagados permanentemente.\n`;
          logData = "";
          resetStats();
          return; // para não processar linhas
        }

        // Processar apenas as novas linhas
        const newLogContent = text.slice(lastSavedLogLength);
        const lines = newLogContent.split('\n').filter(line => line.trim() !== '');

        lines.forEach(line => {
          processLogLine(line);
        });

        logData = text;
        lastSavedLogLength = text.length;

        // Inverter a ordem dos logs
        const allLines = text.split('\n').filter(line => line.trim() !== '');
        const reversedLines = [...allLines].reverse();

        logBox.innerHTML = highlightLogs(reversedLines.join('\n'));
        logBox.scrollTop = 0;

        updateStats();
        updateCharts();
        updateFirewallStatus();
        await checkTrialStatus();

        if (allLines.length > 500) {
          logBox.innerHTML = highlightLogs(reversedLines.slice(0, 500).join('\n'));
        }
      }
    } catch (error) {
      console.error("Error fetching logs:", error);
      if (!logBox.innerHTML.includes("Error loading logs")) {
        logBox.innerHTML = `<span class="log-entry"><span class="timestamp">[${formatTimestamp()}]</span> <span class='error'>Error loading logs: ${error.message}</span></span>\n` + logBox.innerHTML;
      }
    }
  }

  // Delete logs from server and clear display
  async function deleteLogs() {
    if (!confirm("Tem certeza que deseja apagar TODOS os logs permanentemente?")) {
      return;
    }

    try {
      const response = await fetch('delete-logs.php', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ action: 'delete' })
      });

      if (!response.ok) throw new Error("Falha ao apagar logs");

      const result = await response.json();

      if (result.success) {
        logBox.innerHTML = `[${formatTimestamp()}] Todos os logs foram apagados permanentemente.\n`;
        logData = "";
        resetStats();
        return;
      } else {
        throw new Error(result.message || "Erro desconhecido ao apagar logs");
      }
    } catch (error) {
      console.error("Error deleting logs:", error);
      logBox.innerHTML = `[${formatTimestamp()}] <span class='error'>Erro ao apagar logs: ${error.message}</span>\n` + logBox.innerHTML;
    }
  }

  // Export logs to file
  function exportLogs() {
    const blob = new Blob([logBox.innerText], { type: "text/plain" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `fortigate-logs-${new Date().toISOString().replace(/[:.]/g, "-")}.txt`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  }

  // Exit to index.html
  function exitToIndex() {
    window.location.href = 'logout.php';
  }

  // Inicializar gráficos
  function initializeCharts() {
    // Gráfico por hora
    const hourlyCtx = document.getElementById('hourlyChart').getContext('2d');
    window.hourlyChart = new Chart(hourlyCtx, {
      type: 'bar',
      data: {
        labels: Array.from({length: 24}, (_, i) => `${i}h`),
                                   datasets: [{
                                     label: 'Tentativas Diárias',
                                     data: hourlyAttempts,
                                     backgroundColor: 'rgba(238, 49, 36, 0.7)',
                                   borderColor: 'rgba(238, 49, 36, 1)',
                                   borderWidth: 1,
                                   borderRadius: 4
                                   }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: {
            display: false
          },
          tooltip: {
            backgroundColor: 'rgba(30, 30, 30, 0.95)',
                                   titleColor: 'var(--text-primary)',
                                   bodyColor: 'var(--text-secondary)',
                                   borderColor: 'rgba(255, 255, 255, 0.1)',
                                   borderWidth: 1,
                                   padding: 12,
                                   callbacks: {
                                     label: function(context) {
                                       return `Tentativas: ${context.raw}`;
                                     }
                                   }
          }
        },
        scales: {
          y: {
            beginAtZero: true,
            grid: {
              color: 'rgba(255, 255, 255, 0.05)',
                                   drawTicks: false
            },
            ticks: {
              color: '#FFFFFF',
              padding: 8
            }
          },
          x: {
            grid: {
              display: false
            },
            ticks: {
              color: '#FFFFFF',
              padding: 8,
              stepSize: 1,
              precision: 0
            }
          }
        }
      }
    });

    // Gráfico diário
    const dailyCtx = document.getElementById('dailyChart').getContext('2d');
    window.dailyChart = new Chart(dailyCtx, {
      type: 'line',
      data: {
        labels: ['Domingo', 'Segunda', 'Terça', 'Quarta', 'Quinta', 'Sexta', 'Sábado'],
        datasets: [{
          label: 'Tentativas Semanais',
          data: dailyAttempts,
          fill: true,
          backgroundColor: 'rgba(238, 49, 36, 0.1)',
                                  borderColor: 'var(--primary)',
                                  borderWidth: 2,
                                  tension: 0.3,
                                  pointBackgroundColor: 'var(--primary)',
                                  pointRadius: 4,
                                  pointHoverRadius: 6
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: {
            display: false
          },
          tooltip: {
            backgroundColor: 'rgba(30, 30, 30, 0.95)',
                                  titleColor: 'var(--text-primary)',
                                  bodyColor: 'var(--text-secondary)',
                                  borderColor: 'rgba(255, 255, 255, 0.1)',
                                  borderWidth: 1,
                                  padding: 12,
                                  callbacks: {
                                    label: function(context) {
                                      return `Tentativas: ${context.raw}`;
                                    }
                                  }
          }
        },
        scales: {
          y: {
            beginAtZero: true,
            grid: {
              color: 'rgba(255, 255, 255, 0.05)',
                                  drawTicks: false
            },
            ticks: {
              color: '#FFFFFF',
              padding: 8
            }
          },
          x: {
            grid: {
              display: false
            },
            ticks: {
              color: '#FFFFFF',
              padding: 8,
              stepSize: 1,
              precision: 0
            }
          }
        }
      }
    });
  }

  // Event listeners
  pauseBtn.addEventListener("click", () => {
    isPaused = !isPaused;
    pauseBtn.innerHTML = isPaused ? '<i class="fas fa-play"></i> Resume' : '<i class="fas fa-pause"></i> Pause';
  refreshStatus.className = isPaused ? 'refresh-status refresh-paused' : 'refresh-status refresh-active';
  refreshStatus.innerHTML = isPaused ? '<i class="fas fa-pause"></i> Paused' : '<i class="fas fa-circle-notch fa-spin"></i> Active';
  });

  exportBtn.addEventListener("click", exportLogs);
  resetStatsBtn.addEventListener("click", resetStats);
  deleteLogsBtn.addEventListener("click", deleteLogs);
  exitBtn.addEventListener("click", exitToIndex);

  // Initialize
  loadChartData();
  initializeCharts();
  updateStats();
  setupRenewal();

  // Iniciar a atualização de logs
  fetchLogs();
  const intervalId = setInterval(fetchLogs, refreshInterval);
  const statusIntervalId = setInterval(updateFirewallStatus, 5000);
  const trialIntervalId = setInterval(checkTrialStatus, 30000); // Verificar trial a cada 30 segundos

  // Verificar trial status imediatamente
  checkTrialStatus();

  // Reset automático diário e semanal - CORRIGIDO
  let lastCheckedDate = new Date().getDate();
  setInterval(() => {
    const now = new Date();
    const currentDay = now.getDay();
    const currentDate = now.getDate();
    const currentHour = now.getHours();

    // Verifica se mudou o dia
    if (currentDate !== lastCheckedDate) {
      console.log("Dia mudou - resetando gráfico diário");

      // Zera gráfico por hora
      hourlyAttempts = Array(24).fill(0);

      // Se for domingo 00:00-01:00, zera gráfico semanal
      if (currentDay === 0 && currentHour === 0) {
        console.log("Domingo 00:00 - resetando gráfico semanal");
        dailyAttempts = Array(7).fill(0);
      }

      lastResetDate = now;
      lastSavedLogLength = logData.length;

      updateCharts();
      updateStats();
      saveChartData();

      lastCheckedDate = currentDate;
    }
  }, 60000); // verifica a cada 1 minuto

  // Cleanup on page unload
  window.addEventListener("beforeunload", () => {
    saveChartData();
    clearInterval(intervalId);
    clearInterval(statusIntervalId);
    clearInterval(trialIntervalId);
  });
  </script>

</body>
</html>

